/** @file   spawnevent.h
 * @brief   Declaration of SpawnEvent and SpawnItem - classes.
 * @version $Revision: 1.5 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_SPAWNEVENT_H
#define H_WWW_SPAWNEVENT_H

#include "dynamicevent.h"
#include <string>
#include <list>
#include "gameobject.h"
#include "parametertable.h"

namespace WeWantWar {

/** @class  SpawnItem
 * @brief   Baseclass of all the spawnable entities.
 * @author  Tomi Lamminsaari
 *
 * Each object being spawned during the activation of a spawn event is an
 * instance of this class.
 */
class SpawnItem
{
public:
  /** Literal id codes to be used with GetLiteral() method.
   */
  enum LiteralId {
    /** Item's closing tag */
    EClosingTag,
    /** Item's opening tag. */
    EOpeningTag
  };
  
  /** Default constructor.
   */
  SpawnItem();
  
  /** Destructor.
   */
  virtual ~SpawnItem();
  
  /** The method that gets called when the owning spawn event is activated.
   * The implementation of this method must create the new instance of the
   * object that is about to be spawned and add it to the correct object
   * table.
   */
  virtual void Launch() = 0;
  
  /** Reads the contents of this item from given stream. This reads the
   * stream as long as the @c EClosingTag string was read. Each time a parameter
   * has been read we call ReadParameter() method.
   * @param   aIn               The input stream to read from.
   * @return  Zero if operation was successful.
   */
  int Read( std::istream& aIn );
  
  /** Returns the requested literal.
   * @param   aId               The literal id.
   * @return  The string containing the literal.
   */
  virtual std::string GetLiteral( LiteralId aId ) const = 0;
  
protected:
  /** Each time the Read() method encounters a parameter, it calls this method
   * to read its contents. This method supports the following parameters:
   * <code>pos, t, offset, id, angle, flags, route, uid</code>
   * @param   aIn               The input stream.
   * @param   aParameter        The parameter that was just read from the stream
   * @return  0 if successful.
   */
  virtual int ReadParameter( std::istream& aIn, const std::string& aParameter );

public:
  /** Parameters of this item. */
  eng2d::ParameterTable iParameters;
};


/** @class  SpawnEvent
 * @brief   Creates new entities to the playfield.
 * @author  Tomi Lamminsaari
 *
 * With SpawnEvent you can create new GameObjects and other elements to the
 * playfield. The information about the objects being created is loaded
 * from Map Description File.
 *
 * Each instance of @c SpawnEvent can spawn several different elements.
 */
class SpawnEvent : public DynamicEvent
{
public:

  ///
  /// Static members and structures
  /// =============================

  
  /** Spawnable object typecodes.
   */
  enum {
    SPAWN_ANIMATION,
    SPAWN_ATMOSPHERIC_SOUND,
    SPAWN_BONUSITEM,
    SPAWN_LIGHTFENCE,
    SPAWN_PARTICLES,
    SPAWN_TELEPORTER,
    SPAWN_GAMEOBJECT
  };
  
  /** The ID-codes for the animations to spawn.
   */
  enum {
    /** Campfire animation */
    ANIMID_FIRE1,
    /** Fountain animation */
    ANIMID_FOUNTAIN1,
    /** Explosion animation */
    ANIMID_EXPLOSION,
    /** Wrecking floor animation */
    ANIMID_FRACTION1,
    /** Sewerplate animation 1 */
    ANIMID_SEWERPLATE,
    /** Sewerplate animation 2 */
    ANIMID_SEWERPLATE2
  };
  
  /** The ID-codes for the Light Fences
   */
  enum {
    /** Electric fence */
    FENCEID_ELECTRIC
  };
  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructs new Spawn-event.
   */
  SpawnEvent();
  
  /** Destructor.
   */
  virtual ~SpawnEvent();
  
  
  ///
  /// Methods
  /// =======
  
  
  /** Activates the spawn-event. Creates new GameObjects to the playfield.
   */
  virtual void activate();
  
  /** Reimplemented readData-method
   * @param     fin               THe input stream
   * @return    0 if successful, nonzero otherwise.
   */
  virtual int readData( std::istream& fin );
  
  /** Reimplemented update-method.
   */
  virtual void update();
  
  
protected:

  ///
  /// Protected interface
  /// ===================  
  
  /** Reads the data about the Animation to spawn
   * @param     fin               Input stream
   * @return    Pointer to structure that holds the information of this
   *            spawning or null pointer if fails
   */
  SpawnItem* readAnimationData( std::istream& fin );
  
  /** Reads the data about the atmospheric sounds
   * @param     rIn               The input stream
   * @return    Pointer to structure that holds the information of this
   *            spawning or null pointer if fails.
   */
  SpawnItem* readAtmosphericSoundData( std::istream& rIn );
  
  /** Reads the data about the bonus items
   * @param     rIn               The input stream
   * @return    Pointer to structure that holds information of this
   *            spawning or null pointer if fails.
   */
  SpawnItem* readBonusItem( std::istream& rIn );
    
  /** Reads the LightFence - data.
   * @param     rIn               The input stream
   * @return    Pointer to structure holding the information or nullpointer
   *            if failed.
   */
  SpawnItem* readLightFenceData( std::istream& rIn );
  
  /** Reads the Particle data
   * @param     rIn               The input stream
   * @return    Pointer to SpawnItem or nullpointer if fails.
   */
  SpawnItem* readParticleData( std::istream& rIn );
  
  /** Reads the teleporter data
   * @param     rIn               The input stream to read from.
   * @return    Pointer to SpawnItem or nullpointer if failed.
   */
  SpawnItem* readTeleporterData( std::istream& rIn );  
  

  /** Spawns the objects and entities to the playfield.
   */
  void spawnObjects();
  

  /** Returns the spawn item type that matches to given alien name string.
   * @param   aAlienName        Name of the alien.
   * @return  The spawn item type.
   */
  static int alienNameToSpawnType( const std::string& aAlienName );
   
  ///
  /// Members
  /// =======
  
  /** Vector that holds the information about the entities this object
   * should spawn.
   */
  std::list<SpawnItem*> m_spawnObjects;
  
};

};  // end of namespace

#endif // SPAWN_H

/**
 * Version history
 * ===============
 * $Log: spawnevent.h,v $
 * Revision 1.5  2006/07/27 20:28:19  lamminsa
 * SpawnItem class documented.
 *
 * Revision 1.4  2006/03/16 21:11:08  lamminsa
 * Major changes. Almost completely rewritten.
 *
 * Revision 1.3  2006/02/26 07:39:22  lamminsa
 * no message
 *
 * Revision 1.2  2006/02/17 23:09:13  lamminsa
 * no message
 *
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.2  2006-01-10 22:57:23+02  lamminsa
 * SpawnItem has iParameters member.
 *
 * Revision 1.1  2005-11-23 00:23:42+02  lamminsa
 * alienNameToSpawnType() method added.
 *
 * Revision 1.0  2005-11-06 01:17:17+02  lamminsa
 * Initial revision
 *
 */
 
